/**
 * client.c
 *
 * Програма моделює роботу клієнта служби "Луна", що функціонує
 * на базі датаграмних сокетів у комунікаційному домені AF_UNIX.
 * Адреса сервера задається константою SERVER_ADDRESS, адреса клієнта
 * - константою CLIENT_ADDRESS. Зв'язування клієнтського сокета 
 * з локальною адресою виконується за допомогою функції bind().
 * Підготувавши сокет, програма в циклі запитує від користувача
 * слово і передає його серверу; після чого отримує відповідь сервера,
 * виводить її у стандартний потік виведення результатів і запитує 
 * від користувача продовження роботи
 *
 */

#include "myecho.h"

#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>

/*
 * Шаблон адреси клієнта
 */
#define CLIENT_ADDRESS  "/tmp/myecho-client.XXXXXX"

int main()
{
	// Дескриптор клієнтського сокета
	int s;
	// Адреси сервера і клієнта
	struct sockaddr_un serv_addr, clnt_addr;
	// Довжина адреси сервера і адреси клієнта
	socklen_t saddrlen, caddrlen;
	// Буфер для повідомлення
	char buf[BUFSIZE];
	int fd;
    
	// Розміщує в структурі serv_addr адресу сервера,
	// а в змінній saddrlen - її довжину
	memset(&serv_addr, 0, sizeof(serv_addr));
	serv_addr.sun_family = AF_UNIX;
	strcpy(serv_addr.sun_path, SERVER_ADDRESS);
	saddrlen = sizeof(serv_addr.sun_family) + strlen(serv_addr.sun_path);

	// Створює датаграмний сокет в домені AF_UNIX
	s = socket(AF_UNIX, SOCK_DGRAM, 0);
	if (s == -1) {
		perror("Error creating client socket");
		exit(EXIT_FAILURE);
	}

	// Розміщує в структурі clnt_addr свою адресу,
	// а в змінній caddrlen - її довжину
	memset(&clnt_addr, 0, sizeof(clnt_addr));
	clnt_addr.sun_family = AF_UNIX;
	strcpy(clnt_addr.sun_path, CLIENT_ADDRESS);
	fd = mkstemp(clnt_addr.sun_path);
	if (fd == -1) {
		perror("Error creating temporary file for client socket");
		exit(EXIT_FAILURE);
	}
	close(fd);
	caddrlen = sizeof(clnt_addr.sun_family) + strlen(clnt_addr.sun_path);
	// Прив'язує сокет s до локальної адреси
	unlink(clnt_addr.sun_path);
	if (bind(s, (struct sockaddr *) &clnt_addr, caddrlen) == -1) {
		perror("Error binding client socket");
		exit(EXIT_FAILURE);
	}
 
	do {
		// Запитує слово від користувача
		printf("Request: ");
		// Розміщує слово в буфері
		scanf("%s", buf);
		// Передає повідомлення серверу
		if (sendto(s, buf, sizeof(buf), 0, 
		           (struct sockaddr *) &serv_addr, saddrlen) == -1) {
			perror("Error sending datagram to server");
			unlink(clnt_addr.sun_path);
			exit(EXIT_FAILURE);
		}

		// Очищає буфер
		buf[0] = '\0';
		// Отримує відповідь сервера
		if (recv(s, buf, sizeof(buf), 0) == -1) {
			perror("Error receiving datagram from server");
			unlink(clnt_addr.sun_path);
			exit(EXIT_FAILURE);
		}
		// Виводить відповідь сервера
		printf("Answer: %s\n", buf);

		// Запитує від користувача продовження роботи
		printf("Another request? [y/n] ");
		scanf("%s", buf);
	} while (!(strcmp(buf, "y") * strcmp(buf, "Y")));

	unlink(clnt_addr.sun_path);
	exit(EXIT_SUCCESS);
}
